<x-user-layout :title="'Swap'">
    <main class="w-full mx-auto p-4 pb-24 md:pb-4 bg-gray-100 text-gray-900 dark:bg-gray-900 dark:text-white transition-all duration-300">
        <!-- Header -->
        <div class="flex items-center justify-between mb-8">
            <a href="{{ route('dashboard') }}" class="text-gray-600 dark:text-gray-400 hover:text-gray-900 dark:hover:text-white">
                <i class="fas fa-arrow-left text-xl"></i>
            </a>
            <h1 class="text-xl font-semibold text-center flex-1">Swap</h1>
            <div class="w-8"></div>
        </div>

        <form id="swapForm" class="space-y-4">
            @csrf
            <!-- From Section -->
            <div class="bg-white dark:bg-gray-800 rounded-lg p-4">
                <div class="flex justify-between items-center mb-4">
                    <span class="text-gray-400 dark:text-gray-300">From</span>
                    <div id="fromBalance" class="text-sm text-gray-400"></div>
                </div>
                
                <div class="flex items-center justify-between mb-4">
                    <div class="flex items-center cursor-pointer" onclick="openCryptoSelector('from')">
                        <div id="fromCryptoIcon" class="w-10 h-10 rounded-full flex items-center justify-center mr-3 bg-gray-700">
                            <i class="fas fa-coins text-white"></i>
                        </div>
                        <div class="flex items-center">
                            <span id="fromCryptoLabel" class="font-semibold mr-2">Select</span>
                            <i class="fas fa-chevron-right text-gray-400 dark:text-gray-300"></i>
                        </div>
                    </div>
                    <input type="number" 
                           id="fromAmount" 
                           name="amount" 
                           placeholder="0.00"
                           min="0"
                           step="any"
                           class="bg-transparent text-right text-2xl w-32 focus:outline-none"
                           oninput="calculateSwap()">
                </div>

                <div class="flex items-center justify-between mb-4">
                    <div class="flex space-x-2">
                        <button type="button" onclick="setAmount(25)" class="px-3 py-1 rounded-md bg-gray-800 dark:bg-gray-700 text-white text-sm">25%</button>
                        <button type="button" onclick="setAmount(50)" class="px-3 py-1 rounded-md bg-gray-800 dark:bg-gray-700 text-white text-sm">50%</button>
                        <button type="button" onclick="setAmount(75)" class="px-3 py-1 rounded-md bg-gray-800 dark:bg-gray-700 text-white text-sm">75%</button>
                        <button type="button" onclick="setAmount(100)" class="px-3 py-1 rounded-md bg-gray-800 dark:bg-gray-700 text-white text-sm">100%</button>
                    </div>
                    <span id="fromUsdValue" class="text-gray-400 dark:text-gray-300">≈ $0.00</span>
                </div>
            </div>

            <!-- Swap Direction Button -->
            <div class="flex justify-center -my-2 relative z-10">
                <button type="button" onclick="swapDirections()" class="w-10 h-10 rounded-full bg-gray-700 dark:bg-gray-600 flex items-center justify-center">
                    <i class="fas fa-arrow-down text-gray-300 dark:text-gray-400"></i>
                </button>
            </div>

            <!-- To Section -->
            <div class="bg-white dark:bg-gray-800 rounded-lg p-4">
                <div class="flex justify-between items-center mb-4">
                    <span class="text-gray-400 dark:text-gray-300">To</span>
                    <div id="toBalance" class="text-sm text-gray-400"></div>
                </div>
                
                <div class="flex items-center justify-between mb-4">
                    <div class="flex items-center cursor-pointer" onclick="openCryptoSelector('to')">
                        <div id="toCryptoIcon" class="w-10 h-10 rounded-full flex items-center justify-center mr-3 bg-gray-700">
                            <i class="fas fa-coins text-white"></i>
                        </div>
                        <div class="flex items-center">
                            <span id="toCryptoLabel" class="font-semibold mr-2">Select</span>
                            <i class="fas fa-chevron-right text-gray-400 dark:text-gray-300"></i>
                        </div>
                    </div>
                    <input type="number" 
                           id="toAmount" 
                           readonly
                           placeholder="0.00"
                           class="bg-transparent text-right text-2xl w-32 focus:outline-none">
                </div>

                <div class="flex justify-between">
                    <span id="rate" class="text-sm text-gray-400"></span>
                    <span id="toUsdValue" class="text-gray-400 dark:text-gray-300">≈ $0.00</span>
                </div>
            </div>

            <!-- Swap Button -->
            <button type="submit" 
                    id="swapButton"
                    disabled
                    class="w-full bg-yellow-500 hover:bg-yellow-600 text-black font-semibold py-4 rounded-lg transition-colors mt-8 disabled:opacity-50 disabled:cursor-not-allowed">
                Select currencies
            </button>
        </form>

        <!-- Result Modal -->
        <div class="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center hidden z-50" id="resultModal">
            <div class="bg-white dark:bg-gray-900 rounded-lg p-6 w-11/12 max-w-md">
                <div class="text-center">
                    <!-- Success State -->
                    <div id="successState" class="hidden">
                        <div class="w-16 h-16 bg-green-500 rounded-full flex items-center justify-center mx-auto mb-4">
                            <i class="fas fa-check text-2xl text-white"></i>
                        </div>
                        <h3 class="text-xl font-semibold mb-2 text-gray-900 dark:text-gray-100">Swap Successful!</h3>
                        <p class="text-gray-400 mb-6 dark:text-gray-300">Your transaction has been completed successfully.</p>
                    </div>

                    <!-- Error State -->
                    <div id="errorState" class="hidden">
                        <div class="w-16 h-16 bg-red-500 rounded-full flex items-center justify-center mx-auto mb-4">
                            <i class="fas fa-times text-2xl text-white"></i>
                        </div>
                        <h3 class="text-xl font-semibold mb-2 text-gray-900 dark:text-gray-100">Swap Failed</h3>
                        <p id="errorMessage" class="text-gray-400 mb-6 dark:text-gray-300"></p>
                    </div>

                    <!-- Transaction Details -->
                    <div id="transactionDetails" class="space-y-4 hidden">
                        <div class="flex justify-between text-sm">
                            <span class="text-gray-400 dark:text-gray-300">From</span>
                            <span id="resultFromAmount"></span>
                        </div>
                        <div class="flex justify-between text-sm">
                            <span class="text-gray-400 dark:text-gray-300">To</span>
                            <span id="resultToAmount"></span>
                        </div>
                        <div class="flex justify-between text-sm">
                            <span class="text-gray-400 dark:text-gray-300">Rate</span>
                            <span id="resultRate"></span>
                        </div>
                    </div>

                    <button onclick="closeModal()" 
                            class="w-full bg-yellow-500 hover:bg-yellow-600 text-black font-semibold py-3 rounded-lg transition-colors mt-6">
                        Done
                    </button>
                </div>
            </div>
        </div>

        <!-- Crypto Selector Modal -->
        <div class="fixed inset-0 bg-black bg-opacity-50 hidden z-50" id="cryptoSelectorModal">
            <div class="bg-white dark:bg-gray-900 rounded-t-xl fixed bottom-0 left-0 right-0 max-h-[80vh] overflow-y-auto">
                <div class="p-4 border-b border-gray-200 dark:border-gray-700">
                    <div class="flex justify-between items-center">
                        <h3 class="text-lg font-semibold">Select Asset</h3>
                        <button onclick="closeCryptoSelector()" class="text-gray-400 hover:text-gray-500">
                            <i class="fas fa-times"></i>
                        </button>
                    </div>
                </div>
                <div class="p-4" id="cryptoList">
                    <!-- Crypto list will be populated dynamically -->
                </div>
            </div>
        </div>
    </main>

    <script>
        let cryptoData = {
            from: { crypto: '', balance: 0, price: 0 },
            to: { crypto: '', balance: 0, price: 0 }
        };

        let prices = @json($prices);
        let cryptoAsset = @json($cryptoAsset);
        let currentSelector = '';

       const CRYPTO_CONFIG = {
   btc: { icon: '/btc-icon.png', color: 'bg-orange-500' },
   eth: { icon: '/eth-icon.png', color: 'bg-blue-500' },
   usdt_trc20: { icon: '/usdt_trc20-icon.png', color: 'bg-green-500' },
   usdt_bep20: { icon: '/usdt_bep20-icon.png', color: 'bg-green-500' }, 
   usdt_erc20: { icon: '/usdt_erc20-icon.png', color: 'bg-green-500' },
   trx: { icon: '/trx-icon.png', color: 'bg-red-500' },
   bnb: { icon: '/bnb-icon.png', color: 'bg-yellow-500' },
   dot: { icon: '/dot-icon.png', color: 'bg-pink-500' },
   sol: { icon: '/sol-icon.png', color: 'bg-purple-500' },
   bch: { icon: '/bch-icon.png', color: 'bg-green-500' },
   ltc: { icon: '/ltc-icon.png', color: 'bg-blue-500' },
   xlm: { icon: '/xlm-icon.png', color: 'bg-blue-500' },
   dash: { icon: '/dash-icon.png', color: 'bg-blue-500' }
};

        function getDisplayName(crypto) {
            return crypto.split('_').map(part => part.toUpperCase()).join(' ');
        }

        function getPriceForCrypto(crypto) {
            if (crypto.startsWith('usdt_')) return 1; // USDT is always $1
            const priceMap = {
                'btc': 'bitcoin',
                'eth': 'ethereum',
                'trx': 'tron',
                'bnb': 'binancecoin',
                'dot': 'polkadot',
                'sol': 'solana',
                // Add other mappings
            };
            const priceName = priceMap[crypto] || crypto;
            return prices[priceName]?.usd || 0;
        }

        function updateUI() {
            const swapButton = document.getElementById('swapButton');
            if (cryptoData.from.crypto && cryptoData.to.crypto) {
                swapButton.removeAttribute('disabled');
                swapButton.textContent = 'Swap';
            } else {
                swapButton.setAttribute('disabled', 'disabled');
                swapButton.textContent = 'Select currencies';
            }

            calculateSwap();
        }

        function calculateSwap() {
            const fromAmount = parseFloat(document.getElementById('fromAmount').value) || 0;
            if (fromAmount && cryptoData.from.price && cryptoData.to.price) {
                const usdValue = fromAmount * cryptoData.from.price;
                const toAmount = usdValue / cryptoData.to.price;
                
                document.getElementById('fromUsdValue').textContent = `≈ $${usdValue.toFixed(2)}`;
                document.getElementById('toAmount').value = toAmount.toFixed(4);
                document.getElementById('toUsdValue').textContent = `≈ $${usdValue.toFixed(2)}`;
                
                if (cryptoData.from.crypto && cryptoData.to.crypto) {
                    const rate = toAmount / fromAmount;
                    document.getElementById('rate').textContent = 
                        `1 ${getDisplayName(cryptoData.from.crypto)} = ${rate.toFixed(4)} ${getDisplayName(cryptoData.to.crypto)}`;
                }
            } else {
                document.getElementById('fromUsdValue').textContent = '≈ $0.00';
                document.getElementById('toAmount').value = '';
                document.getElementById('toUsdValue').textContent = '≈ $0.00';
                document.getElementById('rate').textContent = '';
            }
        }

        function openCryptoSelector(type) {
            currentSelector = type;
            const modal = document.getElementById('cryptoSelectorModal');
            const cryptoList = document.getElementById('cryptoList');
            cryptoList.innerHTML = '';

            // Get selected cryptos
            const selectedFrom = cryptoData.from.crypto;
            const selectedTo = cryptoData.to.crypto;

            Object.keys(CRYPTO_CONFIG).forEach(crypto => {
                const balanceField = `${crypto}_balance`;
                const statusField = `${crypto}_status`;
                
                if (cryptoAsset[statusField] && 
                    crypto !== (type === 'from' ? selectedTo : selectedFrom)) {
                    
                    const balance = cryptoAsset[balanceField];
                    const config = CRYPTO_CONFIG[crypto];
                    const displayName = getDisplayName(crypto);
                    
                    const item = document.createElement('div');
                    item.className = 'flex items-center justify-between p-4 hover:bg-gray-100 dark:hover:bg-gray-800 cursor-pointer';
                    item.onclick = () => selectCrypto(crypto);
                    
                    item.innerHTML = `
                        <div class="flex items-center">
                            <div class="w-10 h-10 rounded-full flex items-center justify-center mr-3">
                                <img src="${config.icon}" alt="crypto icon" class="w-10 h-10">
                            </div>
                            <div>
                                <div class="font-medium">${displayName}</div>
                                <div class="text-sm text-gray-400">Balance: ${balance}</div>
                            </div>
                        </div>
                    `;
                    
                    cryptoList.appendChild(item);
                }
            });

            modal.classList.remove('hidden');
        }

        function closeCryptoSelector() {
            document.getElementById('cryptoSelectorModal').classList.add('hidden');
            currentSelector = '';
        }

        function selectCrypto(crypto) {
            const config = CRYPTO_CONFIG[crypto];
            const displayName = getDisplayName(crypto);
            const balance = cryptoAsset[`${crypto}_balance`];
            const price = getPriceForCrypto(crypto);

            // Update cryptoData
            cryptoData[currentSelector] = {
                crypto: crypto,
                balance: balance,
                price: price
            };

            // Update UI elements
            document.getElementById(`${currentSelector}CryptoIcon`).className = 
                `w-10 h-10 rounded-full flex items-center justify-center mr-3`;
            document.getElementById(`${currentSelector}CryptoIcon`).innerHTML = 
                `<img src="${config.icon}" alt="crypto icon" class="w-10 h-10">`;
            document.getElementById(`${currentSelector}CryptoLabel`).textContent = displayName;
            document.getElementById(`${currentSelector}Balance`).textContent = 
                `Available: ${balance} ${displayName}`;

            closeCryptoSelector();
            updateUI();
        }

        function swapDirections() {
            const temp = {...cryptoData.from};
            cryptoData.from = {...cryptoData.to};
            cryptoData.to = temp;

            // Swap UI elements
            const fromIcon = document.getElementById('fromCryptoIcon').innerHTML;
            const fromLabel = document.getElementById('fromCryptoLabel').textContent;
            const fromIconClass = document.getElementById('fromCryptoIcon').className;
            const fromBalance = document.getElementById('fromBalance').textContent;

            document.getElementById('fromCryptoIcon').innerHTML = document.getElementById('toCryptoIcon').innerHTML;
            document.getElementById('fromCryptoLabel').textContent = document.getElementById('toCryptoLabel').textContent;
            document.getElementById('fromCryptoIcon').className = document.getElementById('toCryptoIcon').className;
            document.getElementById('fromBalance').textContent = document.getElementById('toBalance').textContent;

            document.getElementById('toCryptoIcon').innerHTML = fromIcon;
            document.getElementById('toCryptoLabel').textContent = fromLabel;
            document.getElementById('toCryptoIcon').className = fromIconClass;
            document.getElementById('toBalance').textContent = fromBalance;

            calculateSwap();
        }

        function setAmount(percentage) {
            if (!cryptoData.from.crypto) return;
            
            const maxAmount = cryptoData.from.balance;
            const amount = (maxAmount * percentage) / 100;
            document.getElementById('fromAmount').value = amount;
            calculateSwap();
        }

        function showSuccess(data) {
            const modal = document.getElementById('resultModal');
            document.getElementById('successState').classList.remove('hidden');
            document.getElementById('errorState').classList.add('hidden');
            document.getElementById('transactionDetails').classList.remove('hidden');
            
            document.getElementById('resultFromAmount').textContent = 
                `${data.from_amount} ${getDisplayName(data.from_crypto)}`;
            document.getElementById('resultToAmount').textContent = 
                `${data.to_amount} ${getDisplayName(data.to_crypto)}`;
            document.getElementById('resultRate').textContent = 
                `1 ${getDisplayName(data.from_crypto)} = ${data.rate.toFixed(8)} ${getDisplayName(data.to_crypto)}`;
            
            modal.classList.remove('hidden');
        }

        function showError(message) {
            const modal = document.getElementById('resultModal');
            document.getElementById('successState').classList.add('hidden');
            document.getElementById('errorState').classList.remove('hidden');
            document.getElementById('transactionDetails').classList.add('hidden');
            document.getElementById('errorMessage').textContent = message;
            modal.classList.remove('hidden');
        }

        function closeModal() {
            document.getElementById('resultModal').classList.add('hidden');
            window.location.reload(); // Refresh to update balances
        }

        // Form submission
        document.getElementById('swapForm').addEventListener('submit', async function(e) {
            e.preventDefault();
            
            if (!cryptoData.from.crypto || !cryptoData.to.crypto) {
                showError('Please select currencies');
                return;
            }

            const amount = parseFloat(document.getElementById('fromAmount').value);
            if (!amount || amount <= 0) {
                showError('Please enter a valid amount');
                return;
            }

            if (amount > cryptoData.from.balance) {
                showError('Insufficient balance');
                return;
            }

            try {
                const response = await fetch('{{ route("swap.process") }}', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': document.querySelector('input[name="_token"]').value
                    },
                    body: JSON.stringify({
                        from_crypto: cryptoData.from.crypto,
                        to_crypto: cryptoData.to.crypto,
                        amount: amount
                    })
                });

                const result = await response.json();
                if (result.success) {
                    showSuccess(result.data);
                } else {
                    showError(result.message);
                }
            } catch (error) {
                console.error('Swap error:', error);
                showError('An error occurred while processing your swap');
            }
        });
    </script>
  
</x-user-layout>